// regdump.cpp

#include <windows.h>
#include <iostream.h>
#include <iomanip.h>

enum {
  SubKeyMax = 80
};

void usage()
{
  cerr << "usage: regdump " << "key-path" << endl;
  exit(-1);
}

// Table listing info on legal root keys:

struct { 
  char* name;
  HKEY handle;
} rk[] = {
  { "HKEY_CLASSES_ROOT", HKEY_CLASSES_ROOT },
  { "HKEY_CURRENT_USER", HKEY_CURRENT_USER },
  { "HKEY_LOCAL_MACHINE", HKEY_LOCAL_MACHINE },
  { "HKEY_USERS", HKEY_USERS },
  { "HKEY_DYN_DATA", HKEY_DYN_DATA },
  { NULL, NULL }
};

/*
  This is a convenience function for the cases where we just have a string
  to put out together with Windows's wisdom on what might have happened.
  Just to be tidy, we strip the trailing CRLF in the Windows error message.
*/

void appendError(ostream& astream, DWORD anerrNo = 0)
{
  enum { MsgLen = 500 };
  DWORD errNo = anerrNo ? anerrNo : GetLastError();
  char* errMsgBuf = "";
  char* msgBuf = new char[MsgLen];
  FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
                NULL, errNo,
                MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                (LPTSTR) &errMsgBuf, 0, NULL);
  char* crLoc = strchr(errMsgBuf, '\r');
  if (crLoc) *crLoc = '\0';
  astream << errMsgBuf;
  LocalFree(errMsgBuf);
}

BOOL isRootKey(HKEY h)
{
  for (int i = 0; rk[i].name != NULL; i++) {
    if (rk[i].handle == h) return TRUE;
  }
  return FALSE;
}

HKEY getRootKey(const char* apath)
{
  for (int i = 0; rk[i].name != NULL; i++) {
    size_t rkiLen = strlen(rk[i].name);
    if (!strnicmp(apath, rk[i].name, rkiLen)) {
      if ((apath[rkiLen] == '\0') || (apath[rkiLen] == '\\')) {
        return rk[i].handle;
      }
    }
  }
  return NULL;
}

HKEY getKey(const char* apath, BOOL amissOK, BOOL areportMiss)
{
  HKEY thisKeyHandle, nextKeyHandle;
  if ((thisKeyHandle = getRootKey(apath)) == NULL) usage();
  char subKeyName[SubKeyMax];
  const char* separatorP = apath + strcspn(apath, "\\");
  while (*separatorP) {
    const char* nextSubKeyP = separatorP + 1;
    size_t subKeyLen = strcspn(nextSubKeyP, "\\");
    memcpy(subKeyName, nextSubKeyP, subKeyLen);
    subKeyName[subKeyLen] = '\0';
    separatorP = nextSubKeyP + subKeyLen;
    DWORD rc = RegOpenKeyEx(thisKeyHandle, subKeyName, 0, KEY_READ,
                            &nextKeyHandle);
    if (rc != ERROR_SUCCESS) {
      if (areportMiss) {
        cout << endl << "*** ";
        appendError(cout, rc);
        cout << endl << apath << endl << "***" << endl;
      }
      if (amissOK) {
        if (!isRootKey(thisKeyHandle)) RegCloseKey(thisKeyHandle);
        return NULL;
      }
      exit(0);
    }
    if (!isRootKey(thisKeyHandle)) RegCloseKey(thisKeyHandle);
    thisKeyHandle = nextKeyHandle;
  }
  return thisKeyHandle;
}

void dumpOneValue(HKEY akeyHandle, DWORD avalueNo, DWORD avalueNameSize,
                  DWORD avalueSize)
{
  DWORD dataType;
  DWORD inputValueSize = avalueSize;
  CHAR* valueName = new CHAR[avalueNameSize];
  BYTE* value = new BYTE[avalueSize];
  LONG rc = RegEnumValue(akeyHandle, avalueNo, valueName, &avalueNameSize,
                         NULL, &dataType, value, &avalueSize);
  DWORD safeVSize = (avalueSize < inputValueSize) ? // Thanks, Microsoft!
    avalueSize : inputValueSize;
  if ((rc == ERROR_SUCCESS) || (rc == ERROR_MORE_DATA)) {
    cout << ((*valueName == '\0') ? "@" : valueName) << " is ";
    switch(dataType)
      {
      case REG_BINARY:
        cout << "REG_BINARY (hex): ";
        {
          DWORD j = 0;          // one beyond end of run of identical bytes
          for (DWORD i = 0; i < safeVSize - 1; i++) {
#if 0
            cout << hex << setw(2) << setfill('0') << WORD(value[i]) << ".";
#else
            for (j = i + 1; (j < safeVSize) && (value[j] == value[i]); j++) {}
            cout << hex << setw(2) << setfill('0') << WORD(value[i]);
            if ((j - i) > 1) {
              cout << "x" << (j - i);
              i = j -1;
            }
            if (j < safeVSize) cout << ".";
#endif
          }
          if (j < safeVSize) {
            cout << hex << setw(2) << setfill('0')
                 << WORD(value[safeVSize - 1]);
          }
        }
        break;
      case REG_DWORD_BIG_ENDIAN:
        cout << "REG_DWORD_BIG_ENDIAN (hex): ";
        {
          for (DWORD i = 0; i < safeVSize - 1; i++) {
            cout << hex << setw(2) << setfill('0') << WORD(value[i]) << ".";
          }
          cout << hex << setw(2) << setfill('0') << WORD(value[safeVSize - 1]);
        }
        break;
      case REG_DWORD_LITTLE_ENDIAN:
        cout << "REG_DWORD_LITTLE_ENDIAN (hex): " << hex << *((DWORD*)value);
        break;
      case REG_EXPAND_SZ:
        cout << "REG_EXPAND_SZ: " << (CHAR*)value;
        break;
      case REG_SZ:
        cout << "REG_SZ: " << (CHAR*)value;
        break;
      case REG_LINK:
        cout << "(REG_LINK)";
        break;
      case REG_MULTI_SZ:
        cout << "REG_MULTI_SZ:";
        {
          for (CHAR* thisValue = (CHAR*)value;
               *thisValue;
               thisValue += (strlen(thisValue) + 1)) {
            cout << endl << "  " << thisValue;
          }
        }
        break;
      case REG_NONE:
        cout << "(REG_NONE)";
        break;
      case REG_RESOURCE_LIST:
        cout << "(REG_RESOURCE_LIST)";
        break;
      default:
        cout << "UNKNOWN DATA TYPE (" << dataType << ")";
        break;
      }
    if (rc == ERROR_MORE_DATA) {
      cout << " (ERROR_MORE_DATA with data size " << inputValueSize
        << " in, " << avalueSize << " out)";
    }
    cout << endl;
  }
  else {
    cout << "(Couldn't dump value due to error: ";
    appendError(cout, rc);
    cout << ")" << endl;
  }
  delete valueName;
  delete value;
  cout << dec;
}

void dumpKey(char* akeyName, BOOL amissOK, BOOL areportMiss)
{
  enum { KeyClassSize = 80 };
  DWORD subKeyNameSize;
  DWORD nSubKeys, nValues, maxSubKeyNameSize, maxValueNameSize;
  DWORD maxValueDataSize, keyClassSize = KeyClassSize;
  CHAR keyClass[KeyClassSize];
  FILETIME lastUpdate;
  HKEY keyHandle = getKey(akeyName, amissOK, areportMiss);
  if (keyHandle == NULL) return;

  if (RegQueryInfoKey(keyHandle, keyClass, &keyClassSize, NULL, &nSubKeys, 
                      &maxSubKeyNameSize, NULL, &nValues, &maxValueDataSize,
                      &maxValueNameSize, NULL, NULL) !=  ERROR_SUCCESS) {
    cerr << "Couldn't query info for key " << akeyName << endl;
    appendError(cerr);
    cerr << endl;
    exit(-1);
  }

  // Because RegQueryInfoKey sometimes underestimates max name or value size,
  // and because one value we found on an NT box was 3750 bytes:
  if (maxValueNameSize < 80) maxValueNameSize = 80;
  if (maxValueDataSize < 4095) maxValueDataSize = 4095;
  if (maxSubKeyNameSize < 80) maxSubKeyNameSize = 80;

  CHAR* subKeyName = new CHAR[maxSubKeyNameSize + 1];

  cout << endl << "Key: " << akeyName << endl;
  if (keyClassSize) cout << "Class: " << keyClass << endl;
  cout << endl;

  if (nValues) {  // show all the values
    for (DWORD valueNo = 0; valueNo < nValues; valueNo++) {
      dumpOneValue(keyHandle, valueNo, maxValueNameSize + 1,
                   maxValueDataSize + 1);
    }
  }

  for (DWORD n = 0; ; n++) { // Show all the subkeys - don't trust nSubKeys
    subKeyNameSize = maxSubKeyNameSize;
    LONG rc = RegEnumKeyEx(keyHandle, n, subKeyName,
                           &subKeyNameSize, NULL, NULL, NULL, &lastUpdate);
    if (rc != ERROR_SUCCESS) {
      if (rc != ERROR_NO_MORE_ITEMS) {
        cerr << "Couldn't query info for subkey " << n << ": ";
        appendError(cerr, rc);
        cerr <<  endl;
      }
      break;
    }
    char* keyName = new char[strlen(akeyName) + strlen(subKeyName) + 3];
    strcpy(keyName, akeyName);
    strcat(keyName, "\\");
    strcat(keyName, subKeyName);
    dumpKey(keyName, amissOK, areportMiss);
    delete keyName;
  }

  // close the key
  if (!isRootKey(keyHandle)) {
    if (RegCloseKey(keyHandle) != ERROR_SUCCESS) {
      cerr << "Unable to close subkey" << endl;
      exit(-1);
    }
  }

  delete subKeyName;
}

/*
  RegGlob: This class generates in its constructor a list of full Registry key
  names matching the pattern argument.  The wildcard syntax is limited to
  subkeys consisting of a single asterisk, meaning any subkey will be accepted;
  there can be zero or more wildcard subkeys.  All memory management is
  buried in the class's code; all the user need do to see all the matching key
  names is call next() until it returns NULL.
*/

class RegGlob {
  struct RGEntry {
    RGEntry* next;
    char* fullKeyName;
    RGEntry(char* afullKeyName) {
      next = NULL;
      fullKeyName = new char[strlen(afullKeyName) + 1];
      strcpy(fullKeyName, afullKeyName);
    }
  };
  RGEntry* first, *current;
  //  void traversePattern(char* apattern, size_t anoffset, char* apartialKeyName);
public:
  RegGlob(char* apattern);
  ~RegGlob();
  char* next();
};

/*
  Here we traverse a key pattern containing one or more asterisk subkey
  segments, dumping every subtree we find that matches.  The first argument
  is the wildcard pattern, the second is the length into the pattern we
  won't backtrack past, and the third is the partial rooted key with wildcards
  removed that we dump.
*/

void traversePattern(char* apattern, size_t anoffset,
                     char* apartialKeyName)
{
  char* here = apattern + anoffset;
  if (!memcmp(here, "\\*\\", 2)) { // We're at a wildcard key segment
    enum { KeyClassSize = 80 };
    DWORD subKeyNameSize;
    DWORD nSubKeys, nValues, maxSubKeyNameSize, maxValueNameSize;
    DWORD maxValueDataSize, keyClassSize = KeyClassSize;
    CHAR keyClass[KeyClassSize];
    FILETIME lastUpdate;
    BOOL missOK =               // Bad GetKey OK if not the first wildcard
      BOOL(strstr(apattern, "\\*\\") < here);
    BOOL reportMiss = BOOL(!missOK);
    HKEY keyHandle = getKey(apartialKeyName, missOK, reportMiss);
    if (keyHandle == NULL) return; // e.g. HKLM\SAM\SAM in NT 4.0
    if (RegQueryInfoKey(keyHandle, keyClass, &keyClassSize, NULL, &nSubKeys, 
                        &maxSubKeyNameSize, NULL, &nValues, &maxValueDataSize,
                        &maxValueNameSize, NULL, NULL) !=  ERROR_SUCCESS) {
      cerr << "Couldn't query info for key " << apartialKeyName << endl;
      cerr << GetLastError() << endl;
      exit(-1);
    }

    // Because RegQueryInfoKey sometimes underestimates max name or value size,
    // and because one value we found on an NT box was 3750 bytes:
    if (maxValueNameSize < 80) maxValueNameSize = 80;
    if (maxValueDataSize < 4095) maxValueDataSize = 4095;
    if (maxSubKeyNameSize < 80) maxSubKeyNameSize = 80;

    CHAR* subKeyName = new CHAR[maxSubKeyNameSize + 1];
    for (DWORD n = 0; ; n++) { // Cover all the subkeys - don't trust nSubKeys
      subKeyNameSize = maxSubKeyNameSize;
      LONG rc = RegEnumKeyEx(keyHandle, n, subKeyName,
                             &subKeyNameSize, NULL, NULL, NULL, &lastUpdate);
      if (rc != ERROR_SUCCESS) {
        if (rc != ERROR_NO_MORE_ITEMS) {
          cerr << "Couldn't query info for subkey " << n << ": " << rc <<  endl;
        }
        break;
      }
      char* newKeyName =
        new char[strlen(apartialKeyName) + strlen(subKeyName) + 3];
      strcpy(newKeyName, apartialKeyName);
      strcat(newKeyName, "\\");
      strcat(newKeyName, subKeyName);
      traversePattern(apattern, anoffset + 2, newKeyName);
      delete newKeyName;
    }

    // close the key
    if (!isRootKey(keyHandle)) {
      if (RegCloseKey(keyHandle) != ERROR_SUCCESS) {
        cerr << "Unable to close subkey" << endl;
        exit(-1);
      }
    }

    delete subKeyName;
  }
  else {                        // Not at a wildcard key segment
    char* nextWildcardSignature = strstr(here, "\\*\\");
    if (nextWildcardSignature == NULL) { // No more wildcards
      char* fullKeyName = new char[strlen(apartialKeyName) + strlen(here) + 1];
      strcpy(fullKeyName, apartialKeyName);
      strcat(fullKeyName, here);
      dumpKey(fullKeyName, TRUE, FALSE);
      delete[] fullKeyName;
    }
    else {                      // Not done with wildcards
      size_t extraLength = nextWildcardSignature - here;
      size_t newLength = strlen(apartialKeyName) + extraLength + 1;
      char* newPartialKeyName = new char[newLength];
      strcpy(newPartialKeyName, apartialKeyName);
      memcpy(newPartialKeyName + strlen(apartialKeyName), apattern + anoffset,
             extraLength);
      newPartialKeyName[newLength - 1] = '\0';
      traversePattern(apattern, anoffset + extraLength, newPartialKeyName);
      delete[] newPartialKeyName;
    }
  }
}

RegGlob::RegGlob(char* apattern)
{
  first = NULL;
  char* asteriskP = strchr(apattern, '*');
  if ((asteriskP != NULL) &&
      (asteriskP[-1] == '\\') &&
      (asteriskP[1] == '\\')) traversePattern(apattern, 0, "");
  else first = new RGEntry(apattern);
  current = first;
}

RegGlob::~RegGlob()
{
  for (current = first; current; ) {
    delete[] current->fullKeyName;
    RGEntry* temp = current->next;
    delete current;
    current = temp;
  }
}

char* RegGlob::next()
{
  char* result = NULL;
  if (current) {
    result = current->fullKeyName;
    current = current->next;
  }
  return result;
}

/*
  We use Windows's command-line-in-one-gulp function because a Registry
  key may contain embedded whitespace.  We just want everything after
  the name of the executable.
*/
int main(int /* argc */, char** /*argv */)
{
  LPTSTR commandLine = GetCommandLine();
  if (commandLine != NULL) {
    char* clp = strchr(commandLine, ' '); // up over exe name
    if (clp != NULL) {
      while (*clp == ' ') clp++; // ...and any spaces.
      char* clpTrailer = clp + strlen(clp) - 1;
      while ((clpTrailer > clp) && (*clpTrailer <= ' ')) {
        *clpTrailer-- = '\0';   // Trim trailing whitespace.
      }
      if (strstr(clp, "\\*\\")) traversePattern(clp, 0, "");
      else dumpKey(clp, TRUE, TRUE);
    }
    else usage();
  }
  else usage();

  return 0;
}
