/*
   NTRARP.C - main module for NT kernel-mode driver NTRARP

   Free software Copyright (c) 1999-2002 Lew Perin

   This KMD allows RARP clients and servers to communicate via Ethernet
   or Token Ring.
*/

/*
   Revision History:

   Version  Date      Reason
   -------  --------  ---------------------------------------------------
   1.00     10/27/99  Forked from version 1.32 of the NTBGPS KMD.
   1.01     03/20/02  Now use Dan Lanciani's technique for NdisFreePacket
                      correctness with NT4/2000/XP.
                      We implement NDIS 4.0 for 2000/XP compatibility but
                      our bind/unbind adapters do nothing and return bad
                      status.
   1.02     06/06/03  Because of Win2K trouble, and because we found by
                      tracing that send completion happens at DISPATCH
                      level, we now use Lanciani's technique on
                      NdisDprFreePacket.
*/

/*
  A note on variable names: It won't take much time reading this code
  to see that we're utterly uninterested in the Microsoft variable
  naming conventions that are so widely seen in 95 and NT driver code.
  We find it usually unnecessary and distracting to be constantly told
  that e.g. a certain variable is a constant ULONG.  One stylistic
  convention we try to use religiously may need pointing out: we
  *always* give function arguments names starting with "a".  We find
  it extremely useful to know at a glance that a variable is or is not
  passed to the function by its caller.  (Thanks, Jim Hollenberg!)
*/

#pragma warning(3:4700)         // Local used w/o being initialized
#pragma warning(3:4701)         // local may be used w/o init

#define DBG 1
#include  "ntrarp.h"

/*
   Locked Data
*/

ProtocolBlock rarpContext;
#if 1
ULONG DbgTraceLevel = DBG_TRACE;
ULONG DbgTraceMask = DBG_RX_ALL;
#else
ULONG DbgTraceLevel = 0;
ULONG DbgTraceMask = 0;
#endif

ULONG osMajor;

/*
  Thanks to Dan Lanciani, here is a way around the facts that for
  the NT4 DDK NdisFreePacket
  - the macro version doesn't free a packet in Win2K/XP;
  - the wrapper version doesn't use the spinlock in NT, causing trouble
    in SMP
*/

VOID myNdisFreePacket(IN PNDIS_PACKET Packet)
{
  if(osMajor <= 4) {
    NdisFreePacket(Packet);
  }
  else {
#undef NdisFreePacket
#pragma warning(disable: 4210)
    VOID NdisFreePacket(IN PNDIS_PACKET Packet);
#pragma warning(default: 4210)
    NdisFreePacket(Packet);
  }
}

#define NdisFreePacket myNdisFreePacket

VOID myNdisDprFreePacket(IN PNDIS_PACKET Packet)
{
  if(osMajor <= 4) {
    NdisDprFreePacket(Packet);
  }
  else {
#undef NdisDprFreePacket
#pragma warning(disable: 4210)
    VOID NdisDprFreePacket(IN PNDIS_PACKET Packet);
#pragma warning(default: 4210)
    NdisDprFreePacket(Packet);
  }
}

#define NdisDprFreePacket myNdisDprFreePacket

/* Initialized Unicode strings for which we disable indirection warning: */

#pragma warning(disable: 4057)
NDIS_STRING rarpName = NDIS_STRING_CONST("rarp");
NDIS_STRING devicePrefix = NDIS_STRING_CONST("\\Device\\");
NDIS_STRING symbolicLinkPrefix = NDIS_STRING_CONST("\\DosDevices\\");
NDIS_STRING tcpLinkageKeyName =
NDIS_STRING_CONST("\\Registry\\Machine\\System"
                  L"\\CurrentControlSet\\Services\\Tcpip\\Linkage");
NDIS_STRING bindValueName = NDIS_STRING_CONST("Bind");
#pragma warning(default: 4057)

/*
  We have stencils for Ethernet and Token Ring
  MAC headers.
*/

RFC894Header ethernetHeaderStencil = {
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0, 0, 0, 0, 0, 0,
  htons(PacketTypeRARP)
};

TokenRingHeader tokenRingHeaderStencil = {
  0x10,                         // data frame
  0x40,                         // LLC frame
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0, 0, 0, 0, 0, 0,
  0xc2, 0x70,                   // single-route broadcast, no more RI data
  0xaa, 0xaa, 3, 0, 0, 0,
  htons(PacketTypeRARP)
};

/*
  Note: We once tested the interaction between compiler, linker, etc. in
  getting static data right by changing NdisMedium802_3 to NdisMediumLocalTalk
  in mediumArray; it caused the havoc we expected!
*/

enum { OurMediaMax = 2 };
NDIS_MEDIUM mediumArray[OurMediaMax] = { // shouldn't be on the stack!
  NdisMedium802_3, NdisMedium802_5
};

/*
   Locked Code
*/

/*
  Get a request entry out of the list and fill in the IRP pointer, if any.
*/

PRarpReqNtry allocateReqNtry(POpenInstance aoiP, PIRP airpP)
{
  PRarpReqNtry result = (PRarpReqNtry)
    NdisInterlockedRemoveHeadList(&aoiP->requestList, &aoiP->requestSpinLock);
  if (result) result->irpP = airpP;
  else RarpTrace("Ran out of request entries\n");
  return result;
}

BOOLEAN prepareMACAddressRequest(POpenInstance aoiP, PNDIS_REQUEST arequestP)
{
  switch(aoiP->u1.medium)
    {
    case NdisMedium802_3:
      arequestP->DATA.QUERY_INFORMATION.Oid = OID_802_3_CURRENT_ADDRESS;
      break;
    case NdisMedium802_5:
      arequestP->DATA.QUERY_INFORMATION.Oid = OID_802_5_CURRENT_ADDRESS;
      break;
    default:
      RarpTrace2("MAC address request for unknown medium: %x\n",
                 aoiP->u1.medium);
      return FALSE;
    }
  arequestP->RequestType = NdisRequestQueryInformation;
  arequestP->DATA.QUERY_INFORMATION.InformationBuffer = aoiP->macAddress;
  arequestP->DATA.QUERY_INFORMATION.InformationBufferLength = MacAddressSize;
  return TRUE;
}

/*
  We need to set the filter on this adapter.
*/

void prepareSetFilterRequest(POpenInstance aoiP, PNDIS_REQUEST arequestP,
                             UINT afilter)
{
  aoiP->packetFilter = afilter;
  arequestP->RequestType = NdisRequestSetInformation;
  arequestP->DATA.SET_INFORMATION.Oid = OID_GEN_CURRENT_PACKET_FILTER;
  arequestP->DATA.SET_INFORMATION.InformationBuffer = &aoiP->packetFilter;
  arequestP->DATA.SET_INFORMATION.InformationBufferLength =
    sizeof(aoiP->packetFilter);
}

/*
  Just a convenience function that'll do all we need to do to finish off
  an IRP in one line of code and return the status.
*/

NTSTATUS finishIrp(IN OUT PIRP airpP, IN NTSTATUS astatus, IN ULONG aninfo)
{
  airpP->IoStatus.Status = astatus;
  airpP->IoStatus.Information = aninfo;
  IoCompleteRequest(airpP, IO_NO_INCREMENT);
  return astatus;
}

/*
   Entry point for Ioctl requests. The following functions are supported:

   DIOC_RarpAwaitRarp: App is ready for a bootp packet.

   DIOC_RarpAwaitAny: App is ready for a packet, any packet!

   DIOC_RarpGetVersion: What version of the driver is this?

   DIOC_RarpGetMacAddress: MAC address of first (only?) netcard, please!

   DIOC_RarpGetHWType: bootp (not NDIS) hardware type of MAC.

   DIOC_RarpDumpDriver: Blow the driver's data into the app's buffer.

   To do: Ioctls to tell driver that the app is only receiving
   RARP queries (or responses.)  This will avoid the server's
   driver instance running out of receive packets on its own
   looped back response.
*/

NTSTATUS RarpIoControl(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  NTSTATUS status = STATUS_SUCCESS;
  UINT lengthNeeded;            // how much buffer space we need to write to
  UINT lengthTransferred = 0;   // how much buffer space we did write to

  RarpTrace2("Ioctl function = %x\n",
             ioslP->Parameters.DeviceIoControl.IoControlCode);

  switch (ioslP->Parameters.DeviceIoControl.IoControlCode) {

  case DIOC_RarpAwaitRarp:
  case DIOC_RarpAwaitAny:
    return finishIrp(airpP, NDIS_STATUS_FAILURE, 0);

  case DIOC_RarpGetVersion:
    lengthNeeded = sizeof(UINT);
    if ((airpP->AssociatedIrp.SystemBuffer == NULL) ||
        (ioslP->Parameters.DeviceIoControl.OutputBufferLength <
         lengthNeeded)) {
      status = STATUS_INVALID_PARAMETER;
    }
    else {
      *((UINT*)airpP->AssociatedIrp.SystemBuffer) =
        NTRARP_Major * 100 + NTRARP_Minor;
      lengthTransferred = lengthNeeded;
    }
    return finishIrp(airpP, status, lengthTransferred);

  case DIOC_RarpGetMacAddress:
    lengthNeeded = MacAddressSize;
    if ((airpP->AssociatedIrp.SystemBuffer == NULL) ||
        (ioslP->Parameters.DeviceIoControl.OutputBufferLength <
         lengthNeeded)) {
      status = STATUS_INVALID_PARAMETER;
    }
    else {
      RtlCopyBytes(airpP->AssociatedIrp.SystemBuffer, oiP->macAddress,
                   MacAddressSize);
      lengthTransferred = lengthNeeded;
    }
    return finishIrp(airpP, status, lengthTransferred);

  case DIOC_RarpGetHWType:
    lengthNeeded = sizeof(UCHAR);
    if ((airpP->AssociatedIrp.SystemBuffer == NULL) ||
        (ioslP->Parameters.DeviceIoControl.OutputBufferLength <
         lengthNeeded)) {
      status = STATUS_INVALID_PARAMETER;
    }
    else {
      switch((NDIS_MEDIUM)(oiP->u1.medium)) {
      case NdisMedium802_3:
        *((UCHAR*)airpP->AssociatedIrp.SystemBuffer) =
          (UCHAR)RarpHWTypeEthernet;
        lengthTransferred = lengthNeeded;
        break;
      case NdisMedium802_5:
        *((UCHAR*)airpP->AssociatedIrp.SystemBuffer) = (UCHAR)RarpHWTypeIEEE;
        lengthTransferred = lengthNeeded;
        break;
      default:
        status = NDIS_STATUS_UNSUPPORTED_MEDIA;
      }
    }
    return finishIrp(airpP, status, lengthTransferred);

  case DIOC_RarpDumpDriver:
    lengthNeeded = sizeof(OpenInstance);
    if ((airpP->AssociatedIrp.SystemBuffer == NULL) ||
        (ioslP->Parameters.DeviceIoControl.OutputBufferLength <
         lengthNeeded)) {
      status = STATUS_INVALID_PARAMETER;
    }
    else {
      RtlCopyBytes(airpP->AssociatedIrp.SystemBuffer, oiP,
                   lengthNeeded);
      lengthTransferred = lengthNeeded;
    }
    return finishIrp(airpP, status, lengthTransferred);

  default:
    return finishIrp(airpP, NDIS_STATUS_FAILURE, 0);
  }

}

/*
  It's the MACs that bind to TCP/IP that we want to bind to.  We find them
  here in the form of a dynamically allocated KEY_VALUE_PARTIAL_INFORMATION
  whose Data member has what we need in the form of a REG_MULTI_SZ array.
  We query the appropriate key twice, the first time just to ascertain the
  size of what we'll allocate and the second time "for real".  If anything
  goes wrong we return NULL.  It's the caller's responsibility to free
  what we return.
*/

PKEY_VALUE_PARTIAL_INFORMATION getTcpBindings(void)
{
  PKEY_VALUE_PARTIAL_INFORMATION result = NULL; // what we return 
  OBJECT_ATTRIBUTES objAttrs;
  NTSTATUS status;
  HANDLE keyHandle;

  InitializeObjectAttributes(&objAttrs, &tcpLinkageKeyName,
                             OBJ_CASE_INSENSITIVE, NULL, NULL);
  status = ZwOpenKey(&keyHandle, KEY_READ, &objAttrs);
  if (!NT_SUCCESS(status)) {
    RarpTrace3("Status of %x opening %ws\n", status, tcpLinkageKeyName.Buffer);
  }
  else {
    ULONG resultLength;
    KEY_VALUE_PARTIAL_INFORMATION valueInfo;

    RarpTrace2("Opened %ws\n", tcpLinkageKeyName.Buffer);

    status = ZwQueryValueKey(keyHandle, &bindValueName,
                             KeyValuePartialInformation, &valueInfo,
                             sizeof(valueInfo), &resultLength);
    if (!NT_SUCCESS(status) && (status != STATUS_BUFFER_OVERFLOW)) {
      RarpTrace2("Status of %x querying key value for size\n", status);
    }
    else {                      // We know how big it needs to be.
      ULONG valueInfoLength = valueInfo.DataLength +
        FIELD_OFFSET(KEY_VALUE_PARTIAL_INFORMATION, Data[0]);
      PKEY_VALUE_PARTIAL_INFORMATION valueInfoP =
        (PKEY_VALUE_PARTIAL_INFORMATION)
        ExAllocatePool(PagedPool, valueInfoLength);
      if (valueInfoP != NULL) {
        status = ZwQueryValueKey(keyHandle, &bindValueName,
                                 KeyValuePartialInformation,
                                 valueInfoP,
                                 valueInfoLength, &resultLength);
        if (!NT_SUCCESS(status)) {
          RarpTrace2("Status of %x querying key value\n", status);
        }
        else if (valueInfoLength != resultLength) {
          RarpTrace3("Querying key value result len = %u \n"
                     "but previous len = %u",
                     resultLength, valueInfoLength);
        }
        else if (valueInfoP->Type != REG_MULTI_SZ) {
          RarpTrace2("Tcpip bind value not REG_MULTI_SZ but %u\n",
                     valueInfoP->Type);
        }
        else {                  // It's OK.
#if DBG
          ULONG i;
          WCHAR* dataP = (WCHAR*)(&valueInfoP->Data[0]);
          RarpTrace("Bind value:\n");
          for (i = 0; *dataP != UNICODE_NULL; i++) {
            UNICODE_STRING macName;
            RtlInitUnicodeString(&macName, dataP);
            RarpTrace3("Mac %u = %ws\n", i, macName.Buffer);
            dataP +=
              (macName.Length + sizeof(UNICODE_NULL)) / sizeof(WCHAR);
          }
#endif // DBG
          result = valueInfoP;
        }
      }
    }
    ZwClose(keyHandle);
  }
  return result;
}

NTSTATUS registerProtocol(PNDIS_HANDLE andisProtocolHandleP)
{
  NDIS40_PROTOCOL_CHARACTERISTICS npc;
  NDIS_STATUS result;

  /*
    Initialize the PROTOCOL characteristics for the call to
    NdisRegisterProtocol.
  */

  NdisZeroMemory(&npc, sizeof(npc));
  npc.MajorNdisVersion = NTRARP_NDIS_MAJOR_VERSION;
  npc.MinorNdisVersion = NTRARP_NDIS_MINOR_VERSION;
  npc.Reserved = 0;
  npc.OpenAdapterCompleteHandler = RarpOpenAdapterCompleteHandler;
  npc.CloseAdapterCompleteHandler = RarpCloseAdapterCompleteHandler;
  npc.SendCompleteHandler = RarpSendCompleteHandler;
  npc.TransferDataCompleteHandler = RarpTransferDataCompleteHandler;
  npc.ResetCompleteHandler = RarpResetCompleteHandler;
  npc.RequestCompleteHandler = RarpRequestCompleteHandler;
  npc.ReceiveHandler = RarpReceiveHandler;
  npc.ReceiveCompleteHandler = RarpReceiveCompleteHandler;
  npc.StatusHandler = RarpStatusHandler;
  npc.StatusCompleteHandler = RarpStatusCompleteHandler;
  npc.BindAdapterHandler = RarpBindAdapterHandler;
  npc.UnbindAdapterHandler = RarpUnbindAdapterHandler;
  npc.Name = rarpName;

  RarpTrace("Calling NdisRegisterProtocol...\n");

  NdisRegisterProtocol(&result, andisProtocolHandleP,
                       (PNDIS_PROTOCOL_CHARACTERISTICS)&npc, sizeof(npc));

  if(result != NDIS_STATUS_SUCCESS) {
    RarpTrace2("NdisRegisterProtocol failed with code 0x%X\n", result);
  }
  else {
    RarpTrace("NdisRegisterProtocol succeeded\n");
  }

  return result;
}

void fillDriverObject(IN PDRIVER_OBJECT adriverObjectP)
{
  adriverObjectP->MajorFunction[IRP_MJ_CREATE] = RarpOpen;
  adriverObjectP->MajorFunction[IRP_MJ_CLOSE] = RarpClose;
  adriverObjectP->MajorFunction[IRP_MJ_READ] = RarpRead;
  adriverObjectP->MajorFunction[IRP_MJ_WRITE] = RarpWrite;
  adriverObjectP->MajorFunction[IRP_MJ_CLEANUP] = RarpCleanup;
  adriverObjectP->MajorFunction[IRP_MJ_DEVICE_CONTROL] = RarpIoControl;
  adriverObjectP->DriverUnload = RarpUnload;
}

/*
  Before we fill in the device extension we make sure it's zeroed so the
  matching "empty" function can be sure any non-NULL dynamically
  allocated pointer should be freed.

  If the MAC name is \Device\Elnk31, the device name will be
  \Device\rarpElnk31 and the symbolic link rarpElnk31.
*/

BOOLEAN fillDeviceExtension(IN OUT PDeviceExtension adevExtP,
                            IN PDEVICE_OBJECT adevObjP,
                            IN PUNICODE_STRING amacNameP,
                            IN PUNICODE_STRING adeviceNameP)
{
  NTSTATUS status;

  NdisZeroMemory(adevExtP, sizeof(DeviceExtension));
  adevExtP->deviceObjectP = adevObjP;
  adevExtP->macName.MaximumLength =
    (USHORT)(amacNameP->Length + sizeof(UNICODE_NULL));
  adevExtP->macName.Buffer =
    ExAllocatePool(NonPagedPool, adevExtP->macName.MaximumLength);
  if (adevExtP->macName.Buffer == NULL) return FALSE;
  RtlAppendUnicodeStringToString(&adevExtP->macName, amacNameP);
  adevExtP->deviceName.MaximumLength =
    (USHORT)(adeviceNameP->Length + sizeof(UNICODE_NULL));
  adevExtP->deviceName.Buffer =
    ExAllocatePool(NonPagedPool, adevExtP->deviceName.MaximumLength);
  if (adevExtP->deviceName.Buffer == NULL) return FALSE;
  RtlCopyUnicodeString(&adevExtP->deviceName, adeviceNameP);
  RarpTrace2("Device name in device extension: %ws\n",
             adevExtP->deviceName.Buffer);
  adevExtP->symbolicLink.MaximumLength =
    (USHORT)(symbolicLinkPrefix.Length + adeviceNameP->Length
             - devicePrefix.Length + sizeof(UNICODE_NULL));
  adevExtP->symbolicLink.Buffer =
    ExAllocatePool(NonPagedPool, adevExtP->symbolicLink.MaximumLength);
  if (adevExtP->symbolicLink.Buffer == NULL) return FALSE;
  RtlAppendUnicodeStringToString(&adevExtP->symbolicLink, &symbolicLinkPrefix);
  RtlAppendUnicodeToString(&adevExtP->symbolicLink,
                           adevExtP->deviceName.Buffer +
                           devicePrefix.Length / sizeof(WCHAR));
  RarpTrace2("Symbolic link: %ws\n", adevExtP->symbolicLink.Buffer);
  status = IoCreateSymbolicLink(&adevExtP->symbolicLink, adeviceNameP);
  if (!NT_SUCCESS(status)) {
    RarpTrace2("IoCreateSymbolicLink status = %x\n", status);
    RtlInitUnicodeString(&adevExtP->symbolicLink, NULL);
    // return FALSE;
  }
  return TRUE;
}

/*
  This convenience function deallocates and zeroes the pointer pointed to
  by the argument only if it isn't already NULL.
*/

void empty(PVOID* memPP)
{
  if (*memPP != NULL) {
    ExFreePool(*memPP);
    *memPP = NULL;
  }
}

void emptyDeviceExtension(IN PDeviceExtension adevExtP)
{
  if (adevExtP->symbolicLink.Buffer != NULL) {
    NTSTATUS status = IoDeleteSymbolicLink(&adevExtP->symbolicLink);
    if (!NT_SUCCESS(status)) {
      RarpTrace2("IoDeleteSymbolicLink status = %x\n", status);
    }
    empty(&adevExtP->symbolicLink.Buffer);
  }
  empty(&adevExtP->macName.Buffer);
  empty(&adevExtP->deviceName.Buffer);
}

/*
  Here we take down everything associated with a device object, including
  what goes with its extension.
*/

void deleteDevice(IN OUT PDEVICE_OBJECT adevObjP)
{
  PDeviceExtension devExtP = (PDeviceExtension)adevObjP->DeviceExtension;
  if (devExtP) emptyDeviceExtension(devExtP);
  IoDeleteDevice(adevObjP);
}

BOOLEAN createDevice(IN OUT PDRIVER_OBJECT adriverObjectP,
                     IN PUNICODE_STRING amacNameP)
{
  NTSTATUS status;
  PDEVICE_OBJECT devObjP;
  UNICODE_STRING deviceName;
  BOOLEAN result = FALSE;

  RarpTrace2("createDevice for MAC %ws\n", amacNameP->Buffer);
  if (RtlCompareMemory(amacNameP->Buffer, devicePrefix.Buffer,
                       devicePrefix.Length) < devicePrefix.Length) {
    return result;
  }

  deviceName.Length = 0;
  deviceName.MaximumLength =
    (USHORT)(amacNameP->Length + rarpName.Length + sizeof(UNICODE_NULL));
  deviceName.Buffer = ExAllocatePool(PagedPool, deviceName.MaximumLength);
  if (deviceName.Buffer != NULL) {
    RtlAppendUnicodeStringToString(&deviceName, &devicePrefix);
    RtlAppendUnicodeStringToString(&deviceName, &rarpName);
    RtlAppendUnicodeToString(&deviceName, amacNameP->Buffer +
                             devicePrefix.Length / sizeof(WCHAR));
    RarpTrace2("Device name: %ws\n", deviceName.Buffer);
    status = IoCreateDevice(adriverObjectP, sizeof(DeviceExtension),
                            &deviceName, FILE_DEVICE_TRANSPORT, 0, FALSE,
                            &devObjP);
    if (NT_SUCCESS(status)) {
      PDeviceExtension devExtP = (PDeviceExtension)devObjP->DeviceExtension;
      if (fillDeviceExtension(devExtP, devObjP, amacNameP, &deviceName)){
        devObjP->Flags |= DO_DIRECT_IO;
        result = TRUE;
      }
      else {
        RarpTrace("Couldn't fill device extension\n");
        deleteDevice(devObjP);
      }
    }
    else RarpTrace2("IoCreateDevice status = %x\n", status);
  }
  if (deviceName.Length > 0) ExFreePool(deviceName.Buffer);
  return result;
}

/*
   Main entry point for NDIS driver; called during initialization

   The registry path argument is bogus for a dynamic load, so we ignore it.
   Instead, we find the lower-edge bindings for TCP/IP in the Registry and
   try to create a transport protocol device for each one.
*/

NTSTATUS DriverEntry(IN PDRIVER_OBJECT adriverObjectP,
                     IN PUNICODE_STRING aregistryPathP)
{
  NTSTATUS result = NDIS_STATUS_FAILURE;
  PKEY_VALUE_PARTIAL_INFORMATION tcpBindingsP;
  ULONG osMinor;

  RarpTrace4("%ws: version %d.%d DriverEntry\n",
             rarpName.Buffer, NTRARP_Major, NTRARP_Minor);

  PsGetVersion(&osMajor, &osMinor, NULL, NULL);
  RarpTrace3("NT version %d.%d\n", osMajor, osMinor);

  tcpBindingsP = getTcpBindings();
  if (tcpBindingsP != NULL) {
    result = registerProtocol(&rarpContext.ndisProtocolHandle);
    if (result == NDIS_STATUS_SUCCESS) {
      WCHAR* bindP = (WCHAR*)(&tcpBindingsP->Data[0]);
      UNICODE_STRING macName;
      fillDriverObject(adriverObjectP);
      for (; *bindP != UNICODE_NULL;
           bindP += (macName.Length + sizeof(UNICODE_NULL)) / sizeof(WCHAR)) {
        RtlInitUnicodeString(&macName, bindP);
        createDevice(adriverObjectP, &macName);
      }
      if (adriverObjectP->DeviceObject == NULL) {
        RarpTrace("No devices created\n");
        result = STATUS_UNSUCCESSFUL;
      }
    }
    else rarpContext.ndisProtocolHandle = 0;
    ExFreePool(tcpBindingsP);
  }
  return result;   
}

VOID RarpUnload(IN PDRIVER_OBJECT adriverObjectP)
{
  NDIS_STATUS status;
  PDEVICE_OBJECT devObjP;
  PDEVICE_OBJECT nextDevObjP;

  RarpTrace("RarpUnload\n");
  for (devObjP = (PDEVICE_OBJECT)adriverObjectP->DeviceObject;
       devObjP != NULL;
       devObjP = nextDevObjP) {
    nextDevObjP = devObjP->NextDevice;
    deleteDevice(devObjP);
  }

  if (rarpContext.ndisProtocolHandle != 0) {
    NdisDeregisterProtocol(&status, rarpContext.ndisProtocolHandle);
  }
  RarpTrace2("NdisDeregisterProtocol status = %x\n", status);
}

/*
  This functionette makes an IRP ready to be submitted to an NDIS wrapper
  function that may complete asynchronously.
*/

void pendIrp(IN OUT PIRP airpP)
{
  IoMarkIrpPending(airpP);
  airpP->IoStatus.Status = STATUS_PENDING;
}

/*
  Here we allocate and populate an OpenInstance, returning a status indicating
  whether we succeeded.  Note that even if we fail we might have allocated
  and partially filled the OpenInstance.
*/

NDIS_STATUS fillOpenInstance(POpenInstance* aoiPP,
                             PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  NDIS_STATUS result = NDIS_STATUS_SUCCESS;
  int i;
  POpenInstance oiP =
     (POpenInstance)ExAllocatePool(NonPagedPool, sizeof(OpenInstance));
  *aoiPP = oiP;
  if (!oiP) return STATUS_INSUFFICIENT_RESOURCES;
  NdisZeroMemory(oiP, sizeof(OpenInstance));
  oiP->deviceObjectP = adeviceObjectP;
  oiP->openOrCloseIrpP = airpP;
  InitializeListHead(&oiP->rcvList); // guarded by Cancel Spinlock
  RarpTrace2("Address of rcvList = %x\n", &oiP->rcvList);
  NdisAllocateSpinLock(&oiP->requestSpinLock);
  InitializeListHead(&oiP->requestList);
  for (i = 0; i < MAX_REQUESTS; i++) {
    NdisInterlockedInsertTailList(&oiP->requestList,
                                  &oiP->requests[i].listElement,
                                  &oiP->requestSpinLock);

  }
  NdisAllocateSpinLock(&oiP->txMacHeaderSpinLock);
  InitializeListHead(&oiP->txMacHeaderList);
  for (i = 0; i < MAX_TRANSMITS; i++) {
    NdisInterlockedInsertTailList(&oiP->txMacHeaderList,
                                  &oiP->txMacHeaders[i].listElement,
                                  &oiP->txMacHeaderSpinLock);
  }
  NdisAllocateSpinLock(&oiP->txArpPacketSpinLock);
  InitializeListHead(&oiP->txArpPacketList);
  for (i = 0; i < MAX_TRANSMITS; i++) {
    NdisInterlockedInsertTailList(&oiP->txArpPacketList,
                                  &oiP->txArpPackets[i].listElement,
                                  &oiP->txArpPacketSpinLock);
  }
  NdisAllocatePacketPool(&result, &(oiP->packetPool), NPackets,
                         sizeof(RarpReserved));
  if (result != NDIS_STATUS_SUCCESS) {
    RarpTrace("Couldn't allocate packet pool\n");
    oiP->packetPool = NULL;
  }
  return result;
}

void emptyOpenInstance(POpenInstance aoiP)
{
  NdisFreeSpinLock(&aoiP->requestSpinLock);
  NdisFreeSpinLock(&aoiP->txMacHeaderSpinLock);
  NdisFreeSpinLock(&aoiP->txArpPacketSpinLock);
  if (aoiP->packetPool) {
    NdisFreePacketPool(aoiP->packetPool);
    aoiP->packetPool = NULL;
  }
}

NTSTATUS RarpOpen(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PDeviceExtension deP = (PDeviceExtension)adeviceObjectP->DeviceExtension;
  POpenInstance oiP;
  NDIS_STATUS status = fillOpenInstance(&oiP, adeviceObjectP, airpP);
  RarpTrace("RarpOpen\n");
  if (NT_SUCCESS(status)) {
    PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
    ioslP->FileObject->FsContext = oiP;
    pendIrp(airpP);
    NdisOpenAdapter(&status, &oiP->openErrorStatus, &oiP->adapterHandle,
                    &oiP->u1.selectedMediumIndex, mediumArray, OurMediaMax,
                    rarpContext.ndisProtocolHandle, (NDIS_HANDLE)oiP,
                    &deP->macName, 0, NULL);
    if (status != NDIS_STATUS_PENDING) {
      RarpOpenAdapterCompleteHandler((NDIS_HANDLE)oiP, status,
                                     oiP->openErrorStatus);
    }
    else RarpTrace("NdisOpenAdapter pending\n");
    return NDIS_STATUS_PENDING;
  }
  else {                        // ran out of resources
    if (oiP){
      emptyOpenInstance(oiP); // ...after allocating OpenInstance
      ExFreePool(oiP);
    }
    return finishIrp(airpP, status, 0);
  }
}

NTSTATUS RarpClose(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  NDIS_STATUS status = NDIS_STATUS_SUCCESS;

  RarpTrace("RarpClose\n");
  oiP->openOrCloseIrpP = airpP;
  pendIrp(airpP);
  NdisCloseAdapter(&status, oiP->adapterHandle);
  if (status != NDIS_STATUS_PENDING) {
    RarpCloseAdapterCompleteHandler((NDIS_HANDLE)oiP, status);
  }
  else RarpTrace("NdisCloseAdapter pending\n");
#if 0
  if (status == NDIS_STATUS_SUCCESS) return NDIS_STATUS_PENDING;
  else return status;
#else
  return NDIS_STATUS_PENDING;
#endif
}

/*
  Whether we are sending a RARP request or a RARP reply, the buffer we are
  presented with is not exactly what we transmit.  It has the destination
  address, which has to be merged with the TX MAC header template, and
  the payload, the [R]ARP packet; they go into two MDLs in the outgoing
  packet.

  There is a fair amount of error checking to make sure we have access to
  buffers from the open instance and MDLs from the nonpaged pool.  If we
  fail to get one of the resources we need, we will have to release the ones
  we did get, but we can depend on the send completion handler to release
  MDLs and their buffers even if we do not call NdisSend.
*/

NTSTATUS RarpWrite(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  NDIS_STATUS status;
  PNDIS_PACKET packetP;
  PTxMacHeaderNtry macHeaderNtryP;
  PTxArpPacketNtry arpPacketNtryP;

  RarpTrace3("RarpWrite w/ write length %u, MDL length %u\n",
             ioslP->Parameters.Write.Length,
             MmGetMdlByteCount(airpP->MdlAddress));
  if (ioslP->Parameters.Write.Length < sizeof(ArpPacket)) {
    return finishIrp(airpP, STATUS_INVALID_PARAMETER, 0);
  }

  NdisAllocatePacket(&status, &packetP, oiP->packetPool);
  if (status != NDIS_STATUS_SUCCESS) {
    RarpTrace2("Packet allocation failure on write: %x\n", status);
    return finishIrp(airpP, status, 0);
  }

  RESERVED(packetP)->irpP = airpP;

  arpPacketNtryP = (PTxArpPacketNtry)
    NdisInterlockedRemoveHeadList(&oiP->txArpPacketList,
                                  &oiP->txArpPacketSpinLock);
  if (arpPacketNtryP) {
    PMDL arpPacketMdlP =
      IoAllocateMdl(&arpPacketNtryP->payload,
                    sizeof(arpPacketNtryP->payload), FALSE, FALSE, NULL);
    PRarpBuf prarpBuf = (PRarpBuf)MmGetSystemAddressForMdl(airpP->MdlAddress);
      if (arpPacketMdlP != NULL) {
        RtlCopyBytes(&arpPacketNtryP->payload, &prarpBuf->ap,
                     sizeof(prarpBuf->ap));
        MmBuildMdlForNonPagedPool(arpPacketMdlP);
        NdisChainBufferAtFront(packetP, arpPacketMdlP);

        macHeaderNtryP = (PTxMacHeaderNtry)
          NdisInterlockedRemoveHeadList(&oiP->txMacHeaderList,
                                        &oiP->txMacHeaderSpinLock);
        if (macHeaderNtryP) {
          ULONG macHeaderLength = (oiP->u1.medium == NdisMedium802_3) ?
            sizeof(RFC894Header) : sizeof(TokenRingHeader);
          PMDL macHeaderMdlP =
            IoAllocateMdl(&macHeaderNtryP->payload, macHeaderLength,
                          FALSE, FALSE, NULL);
          if (macHeaderMdlP != NULL) {
            MmBuildMdlForNonPagedPool(macHeaderMdlP);
            if (ioslP->Parameters.Write.Length >= sizeof(RarpBuf)) {
              RtlCopyBytes(macHeaderNtryP->payload.eth.destination, // TR too
                           prarpBuf->remoteMacAddress,
                           sizeof(prarpBuf->remoteMacAddress));
            }
            NdisChainBufferAtFront(packetP, macHeaderMdlP);
            pendIrp(airpP);
            NdisSend(&status, oiP->adapterHandle, packetP);
            RarpTrace2Mask(DBG_SEND, "NdisSend status = %x\n", status);
          }
          else {
            RarpTrace("Couldn't allocate TX MAC header MDL\n");
            NdisInterlockedInsertTailList(&oiP->txMacHeaderList,
                                          &macHeaderNtryP->listElement,
                                          &oiP->txMacHeaderSpinLock);
            status = STATUS_INSUFFICIENT_RESOURCES;
          }
        }
        else {
          RarpTrace("Ran out of TX MAC header entries\n");
          status = STATUS_INSUFFICIENT_RESOURCES;
        }
      }
      else {
        RarpTrace("Couldn't allocate TX [R]ARP packet MDL\n");
        NdisInterlockedInsertTailList(&oiP->txArpPacketList,
                                      &arpPacketNtryP->listElement,
                                      &oiP->txArpPacketSpinLock);
        status = STATUS_INSUFFICIENT_RESOURCES;
      }
  }
  else {
    RarpTrace("Ran out of TX [R]ARP packet entries\n");
    status = STATUS_INSUFFICIENT_RESOURCES;
  }
  if (status != NDIS_STATUS_PENDING) { // Packet was allocated, at least
    RarpSendCompleteHandler(oiP, packetP, status);
    if (status != NDIS_STATUS_SUCCESS) return status;
  }
  if (status == STATUS_INSUFFICIENT_RESOURCES) {
    return STATUS_INSUFFICIENT_RESOURCES;
  }
  else return NDIS_STATUS_PENDING;
}

/*
  There is no point in accepting a read that lacks the buffer space for
  a RARP packet.  A RARP server will want the extra room for the remote
  MAC address too.
*/

NTSTATUS RarpRead(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  NDIS_STATUS status;
  PNDIS_PACKET packetP;
  KIRQL irql;

  RarpTrace3("RarpRead w/ read length %u, MDL length %u\n",
             ioslP->Parameters.Read.Length,
             MmGetMdlByteCount(airpP->MdlAddress));

  if (MmGetMdlByteCount(airpP->MdlAddress) < sizeof(ArpPacket)) {
    RarpTrace2("Read packet size should be at least %u\n",
               sizeof(ArpPacket));
    return finishIrp(airpP, STATUS_INVALID_PARAMETER, 0);
  }

  NdisAllocatePacket(&status, &packetP, oiP->packetPool);
  if (status != NDIS_STATUS_SUCCESS) {
    RarpTrace2("Packet allocation failure on read: %x\n", status);
    return finishIrp(airpP, status, 0);
  }

  RESERVED(packetP)->irpP = airpP;
  pendIrp(airpP);
  NdisChainBufferAtFront(packetP, airpP->MdlAddress);
  IoAcquireCancelSpinLock(&irql);
  InsertTailList(&oiP->rcvList, &RESERVED(packetP)->listElement);
  IoSetCancelRoutine(airpP, RarpCancelReadIrp); // Now it's cancelable
  IoReleaseCancelSpinLock(irql);
  // STATUS_SUCCESS = BSOD from multiple IRP completion!
  return STATUS_PENDING;        
}

NTSTATUS RarpCleanup(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  PLIST_ENTRY listEntryP;

  RarpTrace("RarpCleanup\n");

  pendIrp(airpP);

  do {
    KIRQL irql;
    IoAcquireCancelSpinLock(&irql);
    listEntryP = (IsListEmpty(&oiP->rcvList)) ? NULL :
      RemoveHeadList(&oiP->rcvList);
    IoReleaseCancelSpinLock(irql);
    if (listEntryP != NULL) {
      PRarpReserved reservedP =
        CONTAINING_RECORD(listEntryP, RarpReserved, listElement);
      PNDIS_PACKET packetP =
        CONTAINING_RECORD(reservedP, NDIS_PACKET, ProtocolReserved);
      RarpTrace("Completing read in Cleanup\n");
      RarpTransferDataCompleteHandler(oiP, packetP, STATUS_CANCELLED, 0);
    }
  } while (listEntryP != NULL);
  return finishIrp(airpP, STATUS_SUCCESS, 0);
}

/*
  After successfully opening the adapter, if it's a real LAN adapter we
  want to get its MAC address so the app can ask us for it later.
  We also need to set its filter so we'll see incoming packets.

  The IRP will be finished off in NdisRequestComplete processing if all
  goes well.
*/

VOID
RarpOpenAdapterCompleteHandler(IN NDIS_HANDLE abindingContext,
                               IN NDIS_STATUS astatus,
                               IN NDIS_STATUS aopenErrorStatus)
{
  OpenInstance* oiP = (OpenInstance*)abindingContext;
  PIRP irpP = oiP->openOrCloseIrpP;

  RarpTrace3("OpenAdapterComplete status / openerror = %x / %x\n",
             astatus, aopenErrorStatus);

  if (astatus == NDIS_STATUS_SUCCESS) {
    PRarpReqNtry macAddressReqNtryP = allocateReqNtry(oiP, NULL);
    PRarpReqNtry setFilterReqNtryP = allocateReqNtry(oiP, irpP);
    if (macAddressReqNtryP && setFilterReqNtryP) {
      if (prepareMACAddressRequest(oiP, &macAddressReqNtryP->request)) {
        NdisRequest(&oiP->macAddressReqStatus, oiP->adapterHandle,
                    &macAddressReqNtryP->request);
        if (oiP->macAddressReqStatus != NDIS_STATUS_PENDING) {
          RarpRequestCompleteHandler((NDIS_HANDLE)oiP,
                                     &macAddressReqNtryP->request,
                                     oiP->macAddressReqStatus);
          
        }
        prepareSetFilterRequest(oiP, &setFilterReqNtryP->request,
                                NDIS_PACKET_TYPE_DIRECTED |
                                NDIS_PACKET_TYPE_BROADCAST);
        NdisRequest(&oiP->setFilterReqStatus, oiP->adapterHandle,
                    &setFilterReqNtryP->request);
        if (oiP->setFilterReqStatus != NDIS_STATUS_PENDING) {
          RarpRequestCompleteHandler((NDIS_HANDLE)oiP,
                                     &setFilterReqNtryP->request,
                                     oiP->setFilterReqStatus);
        }
      }
      else astatus = NDIS_STATUS_UNSUPPORTED_MEDIA; // shouldn't happen!
    }
    else {
      astatus = STATUS_INSUFFICIENT_RESOURCES;
    }
  }
  if (astatus != NDIS_STATUS_SUCCESS) {
    // Never launched an NdisRequest
    finishIrp(irpP, astatus, 0);
    emptyOpenInstance(oiP);
    ExFreePool(oiP);
  }
}

/*
  When NDIS tells us the unbinding of an adapter's complete, we want to
  make sure we don't keep a dangling handle for it.
*/

VOID RarpCloseAdapterCompleteHandler(IN NDIS_HANDLE abindingContext,
                                     IN NDIS_STATUS astatus)
{
  OpenInstance* oiP = (OpenInstance*)abindingContext;
  PIRP irpP = oiP->openOrCloseIrpP;

  RarpTrace2("CloseAdapterCompleteHandler status = %x\n", astatus);

  emptyOpenInstance(oiP);

  finishIrp(irpP, astatus, 0);
  ExFreePool(oiP);
}

/*
  Here, because we need to handle the freeing of MDLs we tried to create
  in the write dispatch routine, we handle all business of that type, even
  if the write dispatch routine ran into trouble before calling NdisSend.
  We have a packet which may have a MAC header in front of an
  ARP packet.  We don't need to dispose of the user data MDL; NT is nice enough
  to do that for us.
*/

VOID RarpSendCompleteHandler(IN NDIS_HANDLE abindingContext,
                             IN PNDIS_PACKET apacketP,
                             IN NDIS_STATUS astatus)
{
  OpenInstance* oiP = (OpenInstance*)abindingContext;
  RarpReserved* reservedP = (RarpReserved*)(apacketP->ProtocolReserved);
  PIRP irpP = reservedP->irpP;
  UINT userBytesSent = (astatus == NDIS_STATUS_SUCCESS) ?
    MmGetMdlByteCount(irpP->MdlAddress) : 0;
  UINT bufferCount;

  RarpTrace2Mask(DBG_SEND, "SendCompleteHandler status = %x\n", astatus);
#pragma warning(disable: 4127)
#pragma warning(disable: 4702)
  NdisQueryPacket(apacketP, NULL, &bufferCount, NULL, NULL);
#pragma warning(default: 4702)
#pragma warning(default: 4127)

  if (bufferCount > 1) {
    PNDIS_BUFFER macHeaderMdlP;
    MacHeaderPayload* macHeaderPayloadP;
    PTxMacHeaderNtry txMacHeaderNtryP;
    NdisUnchainBufferAtFront(apacketP, &macHeaderMdlP);
    macHeaderPayloadP = MmGetSystemAddressForMdl(macHeaderMdlP);
    txMacHeaderNtryP =
      CONTAINING_RECORD(macHeaderPayloadP, TxMacHeaderNtry, payload);
    NdisInterlockedInsertTailList(&oiP->txMacHeaderList,
                                  &txMacHeaderNtryP->listElement,
                                  &oiP->txMacHeaderSpinLock);
  }

  if (bufferCount > 0) {
    PNDIS_BUFFER arpPacketMdlP;
    ArpPacket* arpPacketPayloadP;
    PTxArpPacketNtry txArpPacketNtryP;
    NdisUnchainBufferAtFront(apacketP, &arpPacketMdlP);
    arpPacketPayloadP = MmGetSystemAddressForMdl(arpPacketMdlP);
    txArpPacketNtryP =
      CONTAINING_RECORD(arpPacketPayloadP, TxArpPacketNtry, payload);
    NdisInterlockedInsertTailList(&oiP->txArpPacketList,
                                  &txArpPacketNtryP->listElement,
                                  &oiP->txArpPacketSpinLock);
  }

  NdisReinitializePacket(apacketP);
  if (KeGetCurrentIrql() == DISPATCH_LEVEL) {
    NdisDprFreePacket(apacketP);
  }
  else NdisFreePacket(apacketP);
  finishIrp(irpP, astatus, userBytesSent);
}

/*
   The NDIS wrapper calls this only on an asynchronous transfer, i.e., only
   if our receive handler's call to NdisTransferData() returned with a pending
   status.  In order to centralize completion handling, if our receive
   handler is lucky enough to call NdisTransferData() and not be told to wait,
   it calls this function as if it were the mighty wrapper.

   In fact, since this is where we recycle a receive buffer, we have to
   call this function every time we consume a receive buffer, even if we
   don't use NdisTransferData().  This happens in promiscuous mode.
*/

VOID RarpTransferDataCompleteHandler(IN NDIS_HANDLE abindingContext,
                                     IN PNDIS_PACKET apacketP,
                                     IN NDIS_STATUS astatus,
                                     IN UINT abytesTransferred)
{
  OpenInstance* oiP = (OpenInstance*)abindingContext;
  RarpReserved* reservedP = (RarpReserved*)(apacketP->ProtocolReserved);
  PIRP irpP = reservedP->irpP;

  RarpTrace2Mask(DBG_RECEIVE,
                 "RarpTransferDataCompleteHandler status = %x", astatus);

  NdisReinitializePacket(apacketP); // recycle the packet
  if (KeGetCurrentIrql() == DISPATCH_LEVEL) {
    NdisDprFreePacket(apacketP);
  }
  else NdisFreePacket(apacketP); // Put the packet on the free queue
  if (astatus != STATUS_CANCELLED) finishIrp(irpP, astatus, abytesTransferred);
}

VOID RarpResetCompleteHandler(IN NDIS_HANDLE abindingContext,
                              IN NDIS_STATUS astatus)
{
  RarpTrace("ResetCompleteHandler Entry...\n");
}

VOID RarpRequestCompleteHandler(IN NDIS_HANDLE abindingContext,
                                IN PNDIS_REQUEST andisRequest,
                                IN NDIS_STATUS astatus)
{
  POpenInstance oiP = (POpenInstance)abindingContext;
  PRarpReqNtry reqNtryP = CONTAINING_RECORD(andisRequest, RarpReqNtry,
                                            request);
  PIRP irpP = reqNtryP->irpP;

  RarpTrace2("RequestCompleteHandler status = %x\n", astatus);

  NdisInterlockedInsertTailList(&oiP->requestList, &reqNtryP->listElement,
                                &oiP->requestSpinLock);
  if (irpP) {
    finishIrp(irpP, astatus, 0);
    if (irpP == oiP->openOrCloseIrpP) { // Completing MJ_CREATE
      if (NT_SUCCESS(astatus)) { // Good!
        UINT i;
        oiP->openOrCloseIrpP = 0;
        for (i = 0; i < MAX_TRANSMITS; i++) {
          if (oiP->u1.medium == NdisMedium802_3) {
            RtlCopyBytes(&oiP->txMacHeaders[i].payload.eth,
                         &ethernetHeaderStencil, sizeof(RFC894Header));
            RtlCopyBytes(&oiP->txMacHeaders[i].payload.eth.source,
                         oiP->macAddress, MacAddressSize);
          }
          else {
            RtlCopyBytes(&oiP->txMacHeaders[i].payload.tr,
                         &tokenRingHeaderStencil, sizeof(TokenRingHeader));
            oiP->txMacHeaders[i].payload.tr.source[0] =
              (UCHAR)(oiP->macAddress[0] | TokenRingRoutingInfoIndicator);
            RtlCopyBytes(oiP->txMacHeaders[i].payload.tr.source + 1,
                         oiP->macAddress, MacAddressSize - 1);
          }
        }
      }
      else {                    // Bad; won't need OpenInstance stuff
        emptyOpenInstance(oiP);
        ExFreePool(oiP);
      }
    }
  }
}

PNDIS_PACKET _stdcall getRcvPacket(OpenInstance* aoiP)
{
  PNDIS_PACKET packetP = NULL;
  KIRQL irql;
  PLIST_ENTRY listEntryP;
  PRarpReserved reservedP;

  IoAcquireCancelSpinLock(&irql);
  if (!IsListEmpty(&aoiP->rcvList)) {
    listEntryP = RemoveHeadList(&aoiP->rcvList);
    reservedP = CONTAINING_RECORD(listEntryP, RarpReserved, listElement);
    IoSetCancelRoutine(reservedP->irpP, NULL); // Can't cancel it now.
    packetP  =  CONTAINING_RECORD(reservedP, NDIS_PACKET, ProtocolReserved);
  }
  IoReleaseCancelSpinLock(irql);
  
  return packetP;
}

/*
  Here we handle the packet we may be waiting for.

  First we check to see if the app's really waiting, i.e. if there's an
  event.  Then we make sure it's a recognizable RARP packet.  If so,
  we move the data by NdisTransferData (this is necessary because the
  packet may be longer than the lookahead buffer.)  If there is enough
  room in the buffer for the remote MAC address (a RARP server will want
  this), we copy it into its slot in the RarpBuf.  The processing is
  finished by the TransferDataComplete handler, which we call ourselves
  if the data transfer was finished by the time we got control back from
  NdisTransferData.
*/

UCHAR llcSignature[3] = { 0xaa, 0xaa, 3 };

NDIS_STATUS RarpReceiveHandler(IN NDIS_HANDLE abindingContext,
                               IN NDIS_HANDLE amacReceiveContext,
                               IN PVOID aheaderBuffer,
                               IN UINT aheaderBufferSize,
                               IN PVOID alookaheadBuffer,
                               IN UINT alookaheadBufferSize,
                               IN UINT apacketSize)
{
  USHORT packetType = 0xffff;   // initialized to non-RARP value
  UINT rarpOffset = 0;          // LLC/SNAP will augment this.
  OpenInstance* oiP = (OpenInstance*)abindingContext;

  RarpTrace2Mask(DBG_RX_ALL, "ReceiveHandler packet size = %d\n", apacketSize);

  if (RtlCompareMemory(alookaheadBuffer, llcSignature, sizeof(llcSignature))
      == sizeof(llcSignature)) {
    LLCSNAP* llcSnapP = ((LLCSNAP*)alookaheadBuffer);
    packetType = llcSnapP->packetType;
    rarpOffset = sizeof(LLCSNAP);
  }
  else if (aheaderBufferSize == sizeof(RFC894Header)) {
    packetType = ((RFC894Header*)aheaderBuffer)->packetType;
  }
  else if (!oiP->promiscuous) return NDIS_STATUS_NOT_ACCEPTED;

  if (oiP->promiscuous) {
    return NDIS_STATUS_NOT_ACCEPTED; // Are we going to implement this?
  }
  else if (packetType == htons(PacketTypeRARP)) {
    UINT bytesTransferred = 0;
    PNDIS_PACKET packetP = getRcvPacket(oiP);
    PNDIS_BUFFER firstMdlP;
    UINT totalBufferLength, firstBufferLength;
    UINT sourceOffset = (oiP->u1.medium == NdisMedium802_3) ?
      MacAddressSize : MacAddressSize + 2;
    PRarpBuf firstBufferP;

    if (packetP == NULL) {
      RarpTrace("Ran out of receive packets\n");
      return NDIS_STATUS_NOT_ACCEPTED;
    }

    RarpTraceMask(DBG_RECEIVE, "Received a RARP packet\n");
#pragma warning(disable: 4127 4702)
    NdisGetFirstBufferFromPacket(packetP, &firstMdlP, &firstBufferP,
                                 &firstBufferLength, &totalBufferLength);
#pragma warning(default: 4127 4702)
    if (firstBufferLength >= sizeof(RarpBuf)) { // room for MAC address
#if 0                           // IRQ_NOT_LESS_OR_EQUAL!
      RtlCopyBytes(firstBufferP->remoteMacAddress,
                   ((UCHAR*)aheaderBuffer) + sourceOffset, MacAddressSize);
#else
      PIRP irpP = RESERVED(packetP)->irpP;
      RarpBuf* rarpBufP = (RarpBuf*)MmGetSystemAddressForMdl(irpP->MdlAddress);
      NdisMoveMappedMemory(rarpBufP->remoteMacAddress,
                           ((UCHAR*)aheaderBuffer) + sourceOffset,
                           MacAddressSize);
#endif
    }
    NdisTransferData(&oiP->transferDataStatus,
                     oiP->adapterHandle,
                     amacReceiveContext, rarpOffset,
                     sizeof(ArpPacket), packetP, &bytesTransferred);
    if (oiP->transferDataStatus != NDIS_STATUS_PENDING) {
      RarpTransferDataCompleteHandler(abindingContext,
                                      packetP,
                                      oiP->transferDataStatus,
                                      bytesTransferred);
    }
  } else return NDIS_STATUS_NOT_ACCEPTED;

  return(NDIS_STATUS_SUCCESS);
}

VOID RarpReceiveCompleteHandler(IN NDIS_HANDLE abindingContext)
{
}

VOID RarpStatusHandler(IN NDIS_HANDLE abindingContext,
                       IN NDIS_STATUS ageneralStatus,
                       IN PVOID astatusBuffer,
                       IN UINT astatusBufferSize)
{
  RarpTrace("StatusHandler Entry...\n");
}

VOID RarpStatusCompleteHandler(IN NDIS_HANDLE abindingContext)
{
  RarpTrace("StatusCompleteHandler Entry...\n");
}

/* This function is called with the Cancel spin lock because it's been
   assigned to the IRP using IoSetCancelRoutine.  We have to find this
   IRP's packet on the receive list, remove it, and then recycle it.

   Bug alert: This algorithm is from Art Baker's book.  Both _NT Insider_'s
   series on cancel operations and section 12.3 in the DDK Kernel Mode Driver
   Design Guide's algorithm for drivers without a StartIo routine recommend
   ignoring the input IRP and hitting the first queued IRP with cancel
   set to TRUE.

   Is the routine responsible for setting the Cancel Routine in the IRP
   to NULL as Viscarola and Mason say or is it already NULL as Baker says?
   Baker's right, apparently.  And is the Cancel flag set at entry to this
   routine? Yes.
*/

void RarpCancelReadIrp(IN PDEVICE_OBJECT adeviceObjectP, IN PIRP airpP)
{
  PIO_STACK_LOCATION ioslP = IoGetCurrentIrpStackLocation(airpP);
  POpenInstance oiP = ioslP->FileObject->FsContext;
  BOOLEAN foundIt = FALSE;
  PLIST_ENTRY nextLinkP;

  RarpTrace4("RarpCancelReadIrp IRP %x / cancel flag %d / cancel routine %x\n",
             airpP, airpP->Cancel, airpP->CancelRoutine);

  for (nextLinkP = oiP->rcvList.Flink;
       nextLinkP != &oiP->rcvList;
       nextLinkP = nextLinkP->Flink) {
    PRarpReserved reservedP =
      CONTAINING_RECORD(nextLinkP, RarpReserved, listElement);
    PNDIS_PACKET packetP  =
      CONTAINING_RECORD(reservedP, NDIS_PACKET, ProtocolReserved);
    if (reservedP->irpP == airpP) {
      foundIt = TRUE;
      RemoveEntryList(nextLinkP);
      NdisReinitializePacket(packetP); // recycle the packet
      NdisFreePacket(packetP);  // Put the packet on the free queue
      break;
    }
  }

  IoReleaseCancelSpinLock(airpP->CancelIrql);
  if (foundIt){
    RarpTrace("RarpCancelReadIrp: found IRP\n");
    finishIrp(airpP, STATUS_CANCELLED, 0);
  }
}

VOID
RarpBindAdapterHandler(OUT PNDIS_STATUS astatus,
                       IN  NDIS_HANDLE abindAdapterContext,
                       IN  PNDIS_STRING anadapterName,
                       IN  PVOID asystemSpecific1,
                       IN  PVOID asystemSpecific2)
{
    RarpTrace("RarpBindAdapterHandler\n");
    *astatus = NDIS_STATUS_FAILURE;
}

VOID
RarpUnbindAdapterHandler(OUT PNDIS_STATUS astatus,
                         IN  NDIS_HANDLE aprotocolBindingContext,
                         IN  NDIS_HANDLE anunbindAdapterContext)
{
    RarpTrace("RarpUnbindAdapterHandler\n");
    *astatus = NDIS_STATUS_FAILURE;
}
